;|
   dcldemo.lsp Ver 2.2

   Written by Phillip Norman
   Copyright  2006 by Phillip Norman
   nospam.cadd@phillipnorman.com ;remove "nospam."

   Contributions by Gary Fowler (see below)
   
   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
   WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
   PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.

   October 2006

   Comment by the author:
   "After a flood of questions regarding dcl and lisp in the ADG over a period
    of weeks I was inspired to write this dialog box and lisp demo.
    I am NOT a dcl or lisp expert! And I am sure there is more than one way
    to code for the examples included herewith. So with that said I sincerely
    hope that this demo will be of use to someone. It has at least been a good
    learning exercise for me writing it. Also if anyone can be bothered making
    recommendations for correction of errors or simplifications or improvements
    they will be very welcome. Regards, Phill. PS Thanks to Gary for his input."

   Main Features

   => How to use some common dcl dialog tiles and the lisp to drive them
   => How to use global and local variables with dialog boxes including an automatic variable generator (defun ppn_default_gvar
   => How to prepare dialog box defaults using global and local variables (defun ppn:dcl_default_dcldemo
   => How to make two clusters of radio_buttons behave as one cluster (defun ppn:dcl_complist
   => How to turn tiles on and off based on user input and previous selections (defun ppn:dcl_view
   => How to close a dialog for user input and then return to the dialog (defun ppn:dcl_graphscr
   => How to open a second dialog from the first (original code by Gary Fowler) (defun DMO:ABOUT
   => How to align list items in columns in a list_box (original method by Gary Fowler)
   => How to logically connect two toggles and the necessary lisp code (defun ppn:dcl_ends
   => How to use the system registry to store dialog box screen location (original method by Gary Fowler)
   => How to use widgets in dcl like macros in lisp (original method by Gary Fowler)
   => How to pass arguments or data to lisp macros
   => How to "set" variable names and values from lists and strings
   => How to store and read data from text files (defun ppn:dcl_buildlist_compsct

   Version History	DATE		BY	DESCRIPTION
   V1.1			2005-10-07	PPN	*DMO_ROTANG* var changed to ppn_default_gvar system
   V2.0			2005-10-21	PPN	Dynamic code window added (ppn
   V2.1			2005-10-30	PPN	Un-needed setq removed from ppn:dcl_getpos function
						Un-necessary cond removed from ppn:dcl_buildlist_compsct
						Association list format changed to more conventional dotted pairs ("VIEW" "0") to ("VIEW" . "0")
   V2.2			2006-05-01	PPN	Missing code in (dmo:dcl_code (= KEY "ppn_complist")) repaired
|;

;| ppn_default_gvar
   this function is an automatic variable generator. It creates matching pairs of global and local variables
   as defined by a supplied list
   eg.
   (ppn_default_gvar '(("*DMO_COMPID*" 1 188)
		       ("*DMO_WFID*" 1 188)
		       ("*DMO_COMPOP*" ("VIEW" . "0") ("HATCH" . "1") ("HIDE" . "1") ("ENDS" . "0"))
		      )
   )
   the supplied list is in the form of ("*glb_variablename*" value1 value2 etc)
   or to make a variable in the form of an association list ("*glb_variablename*" ("KEY1" "value1") ("KEY2" "value2") etc)
   The format I use for global variables is always 4 characters followed by an underscore, then any number of
   following characters. All global variables are declared with "*" at beginning and end eg. *DMO_COMPID*
   the *^^^_^^^^^^^* format relates specifically to the (substr (....) 6) call below.
   The function automatically creates a matching variable starting at the character immediately after
   the underscore (the sixth character). This matching variable is intended to be local only and should be
   declared as such. eg. GLOBAL VAR NAME = *DMO_COMPID*, MATCHING LOCAL VAR NAME = COMPID
   These matching pairs of variables allow for effective handling of values within dialog box sessions.
   The global variables are only created if they don't already exist. The matching local variables are
   always created according to the supplied list for use in the dialog box or lisp. This way a user can make changes
   to the data in a dialog box and then click on "Cancel" with out effecting the default or stored data.

   eg. In this program there are 8 different types of steel shapes. I have created a unique global variable
   for each shape and a master global variable to remember which one of the 8 shapes was used last.
   For a wide flange steel shape the following variables are used
   	Shape		=	W12x336(W310x500)
        Master Variable =	*DMO_COMPID*
        Local Variable	=	COMPID
        Global Variable	=	*DMO_WFID*
	Local Variable	=	WFID
	Value		=	(1 188) where 1 is the ID number for Wide Flange shapes and a W12x336 is the 188th W Flange in the list

   1 argument: 1. list of global variables
|;
(defun ppn_default_gvar (GVARLIST)
  (foreach GVAR GVARLIST 									;for each var in supplied list
    (progn
      (if (not (car (atoms-family 1 (list (car GVAR))))) 					;check if var already exists
          (set (read (car GVAR)) (cdr GVAR)) 							;if not, create global variable
      )
      (set (read (substr (car GVAR) 6 (- ( strlen (car GVAR)) 6))) (eval (read (car GVAR)))) 	;create local variable using the *^^^_^^^^^^* (6th) format
      (setq GVAR_LIST (cons (car GVAR) GVAR_LIST)) 						;make a list of variable names for later use when "OK" is pressed
    )
  )
)

;| ppn:dcl_setpos
   stores the dialog box position in the system registry
   2 arguments: 1. name of dialog box 2. Dcl position
|;
(defun ppn:dcl_setpos (DCLNAME DCLPOS)
  (vl-registry-write
    (strcat "HKEY_CURRENT_USER\\Software\\" DCLNAME "\\Controls\\Dialog\\PositionX")
    ""
    (rtos (car DCLPOS) 2 0)
  )
  (vl-registry-write
    (strcat "HKEY_CURRENT_USER\\Software\\" DCLNAME "\\Controls\\Dialog\\PositionY")
    ""
    (rtos (cadr DCLPOS) 2 0)
  )
(princ)
)

;| ppn:dcl_getpos
   gets the dialog box position from the system registry
   1 argument: 1. name of dialog box
|;
(defun ppn:dcl_getpos (DCLNAME / XPOS YPOS)
  (setq XPOS (vl-registry-read (strcat "HKEY_CURRENT_USER\\Software\\" DCLNAME "\\Controls\\Dialog\\PositionX")))
  (setq YPOS (vl-registry-read (strcat "HKEY_CURRENT_USER\\Software\\" DCLNAME "\\Controls\\Dialog\\PositionY")))
  (if (and XPOS YPOS)
      (list (atoi XPOS) (atoi YPOS))
      '(-1 -1)
  )
)

;| ppn:dcl_accept
   this function is called by a dialog box "OK" button
   it resets all the global variables as defined by the list GVAR_LIST
   to the values in their corresponding local variables from the dialog box
   NOTE: I have not used the function "get_tile" anywhere in this demo. Why?
   Well just because I find it better to control dialog box values using the global/local
   functions I have designed. If you were to use get_tile this is where you would put it!
|;
(defun ppn:dcl_accept (DCLNAME)
;  (setq ROTANG (get_tile "ppn_rotang")) ;example of use of get_tile. Not used by this demo because ROTANG is part of the GVAR_LIST below.
  (setq DCLPOS (done_dialog 1))
  (ppn:dcl_setpos DCLNAME DCLPOS)
  (foreach GVAR GVAR_LIST
    (set (read GVAR) (eval (read (substr GVAR 6 (- ( strlen GVAR) 6)))))
  )
)

;| ppn:dcl_options
   redefines a specific part of a local association list variable according to dialog box input.
   It is usually part of an (action_tile) function call
   3 arguments: 1. Key of the association list part 2. New value to set 3. local variable name
|;
(defun ppn:dcl_options (KEY VALUE VAR)
  (set VAR (subst (cons KEY VALUE) (assoc KEY (eval VAR)) (eval VAR)))
)

;| ppn:dcl_view
   changes the mode of certain dialog box tiles according to the value of a specific tile
   (action_tile "ppn_view" ...
   and resets the local variable accordingly
   2 arguments: 1. the value of the tile 2. the variable to reset
|;
(defun ppn:dcl_view (VALUE VAR)
  (cond
    ((= VALUE "0")							;first item in list is item "0". In this case "Section"
     (mode_tile "ppn_hatch" 0)						;turn on the hatch toggle
     (mode_tile "ppn_hide" 1)						;turn off the hide lines button
     (mode_tile "ppn_1stend" 1)						;turn off the elevation end line toggle
     (mode_tile "ppn_2ndend" 1)						;turn off the elevation end line toggle
    )
    ((= VALUE "1")
     (mode_tile "ppn_hatch" 1)						;as above but for "Plan"
     (mode_tile "ppn_hide" 0)
     (mode_tile "ppn_1stend" 1)
     (mode_tile "ppn_2ndend" 1)
    )
    ((= VALUE "2")							;as above but for "Elevation"
     (mode_tile "ppn_hatch" 1)
     (mode_tile "ppn_hide" 0)
     (mode_tile "ppn_1stend" 0)
     (mode_tile "ppn_2ndend" 0)
    )
  )
  (ppn:dcl_options "VIEW" VALUE VAR)					;reset the local options variable "VIEW" data
)

;| ppn:dcl_ends
   action_tile function for the logical pairing of two toggles.
   values can be	"off" - "off" 	= 0
	  		"on" - "off"	= 1
  			"off" - "on"	= 2
  			"on" - "on"	= 3
   changes the value of a local variable accordingly
   3 arguments: 1. Key of the association list part 2. New value to set 3. local variable name
|;
(defun ppn:dcl_ends (KEY VALUE VAR)
  (cond									;cond evaluates only the first true expression
    ((= KEY "ppn_1stend")						;if the toggle picked was the 1stend toggle
     (if (= VALUE "1")							;if it was turned on
       (setq VALUE (1+ (atoi (cdr (assoc '"ENDS" COMPOP)))))		;add 1 to the current value of the "ENDS" option
       (setq VALUE (1- (atoi (cdr (assoc '"ENDS" COMPOP)))))		;else subtract 1 from the current value
     )
    )
    ((= KEY "ppn_2ndend")						;if the toggle picked was the 2ndend toggle
     (if (= VALUE "1")							;as above...
       (setq VALUE (+ (atoi (cdr (assoc '"ENDS" COMPOP))) 2))
       (setq VALUE (- (atoi (cdr (assoc '"ENDS" COMPOP))) 2))
     )
    )
  )
  (ppn:dcl_options "ENDS" (itoa VALUE) VAR)				;reset the local options variable "ENDS" data
)

;| ppn:dcl_compsct
   list_box (action_tile "ppn_complist"... function.
   sets the steel shape local variable according to value from the list_box
   uses local variable COMPID to store shape type and exact description
   eg.	for wide flange W12x336(W310x500)
	local variable	=	WFID
	value		=	(1 188)
|;
(defun ppn:dcl_compsct ()
  (setq COMPID (list (car COMPID) COMP#))				;set the local master variable to the current shape type and new shape size
  (cond
    ((= (car COMPID) 1)							;if shape type is wide flange then reset that local variable etc.
     (setq WFID (list 1 COMP#))
    )
    ((= (car COMPID) 2)							;as above for welded wide flange
     (setq WWFID (list 2 COMP#))
    )
    ((= (car COMPID) 3)							;standard channels
     (setq CHID (list 3 COMP#))
    )
    ((= (car COMPID) 4)							;misc. channels
     (setq MCHID (list 4 COMP#))
    )
    ((= (car COMPID) 5)							;angle shapes (ea and ua)
     (setq EAID (list 5 COMP#))
    )
    ((= (car COMPID) 6)							;standard beams
     (setq SBID (list 6 COMP#))
    )
    ((= (car COMPID) 7)							;rhs shapes
     (setq RHSID (list 7 COMP#))
    )
    ((= (car COMPID) 8)							;chs shapes
     (setq CHSID (list 8 COMP#))
    )
  )
)

;| ppn:dcl_complist
   radio_button (action_tile "ppn_wf"... for steel shape types
   resets the list_box "ppn_complist" according to the type of shape
   selected by the radio button.
   also controls two radio_columns as if the were one!
   and resets the local variable COMPID to the new shape
   eg.	for wide flange
	(action_tile "ppn_wf" "(ppn:dcl_complist 1 \"WFID\")")
   2 arguments: 1. the ID code number for the shape type 2. the variable name for the shape type
|;
(defun ppn:dcl_complist (ID VAR)
  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))		;redefine the list for the selected shape type
  (setq COMPID (list ID (cadr (eval (read VAR)))))			;reset the local master variable for the new shape type
  (start_list "ppn_complist")						;reset the list in the dialog box
  (mapcar 'add_list DMO_COMPLIST)
  (end_list)
  (set_tile "ppn_complist" (rtos (cadr (eval (read VAR)))))		;set the correct shape size in the list based on the shape type argument supplied
  (cond
    ((member ID '(1 3 5 7))						;if the radio button picked is in the first column (ID comes from action_tile)
     (foreach STR '("ppn_wwf" "ppn_mch" "ppn_sb" "ppn_chs")		;turn off the buttons in the second column
       (set_tile STR "0")
     )
    )
    ((member ID '(2 4 6 8))						;if the radio button picked is in the second column
     (foreach STR '("ppn_wf" "ppn_ch" "ppn_ea" "ppn_rhs")		;turn off the buttons in the first column
       (set_tile STR "0")
     )
    )
  )
)

;| ppn:dcl_buildlist_compsct
   creates the list of steel shapes for use in list_box "ppn_complist"
   data is stored in tab delimited text files with names in metric and
   imperial format
   2 arguments: 1. name of file for shape type 2. metric or imperial units
|;
(defun ppn:dcl_buildlist_compsct (ID UNITS / PATH FN FNAME COMP_NAME COMP_LIST)
  (setq FNAME (nth (1- ID) '("d_CISC_W.txt" "d_CISC_WWF.txt" "d_CISC_C.txt" "d_CISC_MC.txt"
                             "d_CISC_L.txt" "d_CISC_S.txt" "d_CISC_TS.txt" "d_CISC_HS.txt"))
  )
  (setq FN (findfile FNAME))
  (setq FN (open FN "r"))
  (while (setq COMP_NAME (read-line FN))				;while there are lines of data in the file
    (if (= UNITS 0)							;if units are imperial
      (setq COMP_NAME (last (read (strcat "(" COMP_NAME ")" )))) 	;read the last column in each line
      (setq COMP_NAME (car (read (strcat "(" COMP_NAME ")" ))))		;otherwise read the first column
    )
    (setq COMP_LIST (cons COMP_NAME COMP_LIST))				;put all the shape names in a list
  )
  (close FN)
  (setq COMP_LIST (reverse COMP_LIST))					;put the list in the right order
)

;| ppn:dcl_graphscr
   (action_tile "ppn_graphscr"..... function
   example function for exiting and returning to a dialog box
   for user input from the graphics screen
   No other specific purpose.
|;
(defun ppn:dcl_graphscr ()
  (setq INSP (getpoint "\nSelect an insertion point: "))		;in this example the user is asked to pick an insertion point
(princ)									;basically you can do anything you want but I recommend
)									;keeping it simple

;| ppn:dcl_code
   function to display and hide the lisp code list box
   by changing the dialog box definition used by the
   main program
 |;
(defun ppn:dcl_code ()
  (setq CODE (boole 6 CODE 1))						;logical function changes a value from 1 to 0, or 0 to 1
  (if (= CODE 1)
    (setq DCLNAME "DCLDEMO_CODE")					;use dcl definition for dialog box with code window
    (setq DCLNAME "DCLDEMO")						;use dcl definition for dialog box without code window
  )
  (setq DCLPOS (done_dialog 4))						;close the dialog box remembering the current screen position
)

;;;==============================================================

;| ppn:dcl_default_dcldemo
   dialog box default value function. Defines the global variables needed for the dialog box and other default values
   Uses the global/local variable pairing system as described above in function "ppn_default_gvar"
|;
(defun ppn:dcl_default_dcldemo ()
  (setq UNITS (getvar "measurement"))								;check if drawing is metric or imperial
  (ppn_default_gvar '(("*DMO_COMPID*" 1 188)							;set global default and local vars for
		      ("*DMO_WFID*" 1 188)							;this dialog box
		      ("*DMO_WWFID*" 2 55)
		      ("*DMO_CHID*" 3 5)
		      ("*DMO_MCHID*" 4 18)
		      ("*DMO_EAID*" 5 68)
		      ("*DMO_SBID*" 6 20)
		      ("*DMO_RHSID*" 7 85)
		      ("*DMO_CHSID*" 8 31)
                      ("*DMO_ROTANG*" "0")
		      ("*DMO_COMPOP*" ("VIEW" . "0") ("HATCH" . "1") ("HIDE" . "1") ("ENDS" . "0"))	;variable for dialog box options
                     )										;end of list of variables to declare
  )
  (setq DCLPOS (ppn:dcl_getpos "DCLDEMO")							;set the default dialog box screen position using ppn:dcl_getpos to read the system registry
        DCLNAME "DCLDEMO"									;set the name of the dcl to display (only realy neaded here by the ppn:dcl_code feature)
        INSP '(0 0 0)										;default value for the "ppn_insp" insertion point tile
        CODE 0											;forces the "ppn_code" toggle tile to be off by default
        CODETXT (list "Pick on a button to display its code!")					;default value for the code window when first turned on
  )												;end setq
)												;end defaults

;;;==============================================================
;;; Main program
;;;

(defun c:ppn_dcldemo ( / DMO_COMPLIST GVAR_LIST TILE_COMP COMPID WFID WWFID CHID MCHID EAID SBID
                         RHSID CHSID COMPOP COMP# ENDS1 ENDS2 DMO_DCL_ID DCLPOS DCLNAME COMP_NAME
                         WHAT_NEXT UNITS INSP ROTANG CODE CODETXT)				;define main function and declare local vars
  (setq DMO_DCL_ID (load_dialog "dcldemo06 VER 2.0.dcl"))					;load dialog box definition file
  (ppn:dcl_default_dcldemo)									;set dialog box defaults
  (setq WHAT_NEXT 3)										;establish start value for while loop for open/close dialog for user input
  (while (< 1 WHAT_NEXT)									;start while loop for dialog box control using WHAT_NEXT as a test
    (if (not (new_dialog DCLNAME DMO_DCL_ID "" DCLPOS)) (exit))					;load specific dialog box
    (setq COMP# (cadr COMPID))									;set the current steel shape using LOCAL variable
    (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct (car COMPID) UNITS)
          TILE_COMP (nth (1- (car COMPID)) '("ppn_wf" "ppn_wwf" "ppn_ch" "ppn_mch"
                                             "ppn_ea" "ppn_sb" "ppn_rhs" "ppn_chs"))
    )
;|following cond superceded by the setq above
    (cond											;use cond to determine the current steel shape type using LOCAL variable
      ((= (car COMPID) 1)									;wide flange
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 1 UNITS) TILE_COMP "ppn_wf")		;set the appropriate list of shapes and set a var to use in the dialog
      )
      ((= (car COMPID) 2)									;welded wide flange
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 2 UNITS) TILE_COMP "ppn_wwf")
      )
      ((= (car COMPID) 3)									;standard channels
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 3 UNITS) TILE_COMP "ppn_ch")
      )
      ((= (car COMPID) 4)									;misc. channels
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 4 UNITS) TILE_COMP "ppn_mch")
      )
      ((= (car COMPID) 5)									;angle shapes (ea and ua)
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 5 UNITS) TILE_COMP "ppn_ea")
      )
      ((= (car COMPID) 6)									;standard beams
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 6 UNITS) TILE_COMP "ppn_sb")
      )
      ((= (car COMPID) 7)									;rhs shapes
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 7 UNITS) TILE_COMP "ppn_rhs")
      )
      ((= (car COMPID) 8)									;chs shapes
       (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct 8 UNITS) TILE_COMP "ppn_chs")
      )
    )
|;
    (start_list "ppn_complist")									;fill the list of shapes with the items in variable DMO_COMPLIST
    (mapcar 'add_list DMO_COMPLIST)
    (end_list)
    (set_tile TILE_COMP "1")									;set the radio button tile according to the cond above for the type of shape
    (cond											;determine which tiles to set according to current LOCAL view option
      ((= (cdr (assoc '"VIEW" COMPOP)) "0")							;section
       (mode_tile "ppn_hatch" 0)
       (mode_tile "ppn_hide" 1)
       (mode_tile "ppn_1stend" 1)
       (mode_tile "ppn_2ndend" 1)
      )
      ((= (cdr (assoc '"VIEW" COMPOP)) "1")							;plan
       (mode_tile "ppn_hatch" 1)
       (mode_tile "ppn_hide" 0)
       (mode_tile "ppn_1stend" 1)
       (mode_tile "ppn_2ndend" 1)
      )
      ((= (cdr (assoc '"VIEW" COMPOP)) "2")							;elevation
       (mode_tile "ppn_hatch" 1)
       (mode_tile "ppn_hide" 0)
       (mode_tile "ppn_1stend" 0)
       (mode_tile "ppn_2ndend" 0)
      )
    )
    (cond											;determine which toggles to turn on according to the current LOCAL option
      ((= (cdr (assoc '"ENDS" COMPOP)) "0")							;for elevation end lines
       (setq ENDS1 "0" ENDS2 "0")
      )
      ((= (cdr (assoc '"ENDS" COMPOP)) "1")
       (setq ENDS1 "1" ENDS2 "0")
      )
      ((= (cdr (assoc '"ENDS" COMPOP)) "2")
       (setq ENDS1 "0" ENDS2 "1")
      )
      ((= (cdr (assoc '"ENDS" COMPOP)) "3")
       (setq ENDS1 "1" ENDS2 "1")
      )
    )
    (set_tile "ppn_insp" (strcat "(" (rtos (car insp) 2 0) "," (rtos (cadr insp) 2 0) "," (rtos (caddr insp) 2 0) ")")) ;format the point list for display as text
    (set_tile "ppn_view" (cdr (assoc '"VIEW" COMPOP)))							;set the view tile according to the LOCAL variable
    (set_tile "ppn_rotang" (car ROTANG))								;set the rotation angle tile
    (set_tile "ppn_hide" (cdr (assoc '"HIDE" COMPOP)))							;set the hide lines toggle according to the LOCAL variable
    (set_tile "ppn_hatch" (cdr (assoc '"HATCH" COMPOP)))						;set the hatch toggle according to the LOCAL variable
    (set_tile "ppn_1stend" ENDS1)									;set the end lines toggle according to the cond above
    (set_tile "ppn_2ndend" ENDS2)									;set the end lines toggle according to the cond above
    (set_tile "ppn_complist" (itoa (cadr COMPID)))							;set the steel shape list according to the LOCAL variable
    (set_tile "ppn_code" (itoa CODE))
    (start_list "ppn_codetxt")
    (mapcar 'add_list CODETXT)
    (end_list)                  ;(dmo:dcl_code $key) is only needed for this demo porgram. It is not needed in a normal dcl setup
    (action_tile "ppn_complist" "(dmo:dcl_code $key)(setq COMP# (atoi $value)) (ppn:dcl_compsct)")	;define actions for specific dialog box tiles
    (action_tile "ppn_wf"       "(dmo:dcl_code $key)(ppn:dcl_complist 1 \"WFID\")")			;radio button action
    (action_tile "ppn_wwf"      "(dmo:dcl_code $key)(ppn:dcl_complist 2 \"WWFID\")")			;radio button action
    (action_tile "ppn_ch"       "(dmo:dcl_code $key)(ppn:dcl_complist 3 \"CHID\")")			;radio button action
    (action_tile "ppn_mch"      "(dmo:dcl_code $key)(ppn:dcl_complist 4 \"MCHID\")")			;radio button action
    (action_tile "ppn_ea"       "(dmo:dcl_code $key)(ppn:dcl_complist 5 \"EAID\")")			;radio button action
    (action_tile "ppn_sb"       "(dmo:dcl_code $key)(ppn:dcl_complist 6 \"SBID\")")			;radio button action
    (action_tile "ppn_rhs"      "(dmo:dcl_code $key)(ppn:dcl_complist 7 \"RHSID\")")			;radio button action
    (action_tile "ppn_chs"      "(dmo:dcl_code $key)(ppn:dcl_complist 8 \"CHSID\")")			;radio button action
    (action_tile "ppn_view"     "(dmo:dcl_code $key)(ppn:dcl_view $value 'COMPOP)")			;popup list action
    (action_tile "ppn_rotang"   "(setq ROTANG (list $value))")						;edit box action for rotation angle
    (action_tile "ppn_hide"     "(dmo:dcl_code $key)(ppn:dcl_options \"HIDE\" $value 'COMPOP)")		;toggle action
    (action_tile "ppn_hatch"    "(dmo:dcl_code $key)(ppn:dcl_options \"HATCH\" $value 'COMPOP)")	;toggle action
    (action_tile "ppn_1stend"   "(dmo:dcl_code $key)(ppn:dcl_ends $key $value 'COMPOP)")		;toggle action
    (action_tile "ppn_2ndend"   "(dmo:dcl_code $key)(ppn:dcl_ends $key $value 'COMPOP)")		;toggle action
    (action_tile "ppn_graphscr" "(dmo:dcl_code $key)(setq DCLPOS (done_dialog 2))")			;button to close dialog box temporarily for user input
													;note the (done_dialog 2) that keeps the while loop active
    (action_tile "ppn_about"    "(dmo:dcl_code $Key)(DMO:ABOUT)")					;orignal DMO:ABOUT by Gary Fowler
    (action_tile "ppn_code"     "(dmo:dcl_code $key)(ppn:dcl_code)")
    (action_tile "ppn_repos"    "(dmo:dcl_code $key)(done_dialog 3)(setq DCLPOS '(-1 -1))")		;button to close dialog box temporarily and center it
    (action_tile "accept"       "(ppn:dcl_accept \"DCLDEMO\")")						;ok button action with call to "ppn:dcl_accept" (see above)
    (action_tile "cancel"       "(done_dialog 0)")							;close the dialog box and do nothing (done_dialog 0)
    (setq WHAT_NEXT (start_dialog))									;start the dialog box. Values returned by done_dialog are stored in WHAT_NEXT
    (if (= WHAT_NEXT 2)											;"if" statement to call function after Graphics Screen button is pressed
        (ppn:dcl_graphscr)
    )
  )
  (if (= WHAT_NEXT 1)											;main body of program. this is where you would put anything you want
      (progn												;the program to do after the dialog box is closed by "OK/accept"
        (unload_dialog DMO_DCL_ID)
        (setq GVAR_LIST (reverse GVAR_LIST))
        (foreach GVAR GVAR_LIST (princ (strcat "\n" GVAR " = ")) (prin1 (eval (read GVAR))))		;more than just printing a list of variables you can use them
        (princ "\n.....Yippee. It Worked.....")								;to control your program eg. if you want to know whether to draw
      )													;a section, from the possible choices of section, plan or elevation
  )													;use (= (cdr (assoc '"VIEW" *DMO_COMPOP*)) "0"). This exprssion will be true
(princ)													;if "Section" was selected.
)
(princ)

;|   the following code was originally supplied by Gary Davidson Fowler
     and has been modified for this dialog box demonstration program by Phillip Norman.

     Original, un-modified code is copyright as follows:
     
     Copyright  2004 by Gary Davidson Fowler - Architect
     Richardson, Texas
     email: arch_program@hotmail.com

     Permission to use, copy, modify, and distribute this software
     for any purpose and without fee is hereby GRANTED...Enjoy

     Gary Davidson Fowler PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.  
     Gary Davidson Fowler SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
     MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. Gary Davidson Fowler 
     DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
     UNINTERRUPTED OR ERROR FREE.

     Use, duplication, or disclosure by the U.S. Government is subject to 
     restrictions set forth in FAR 52.227-19 (Commercial Computer 
     Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
     (Rights in Technical Data and Computer Software), as applicable. 
|;
(defun ANIIT ( / col push per)
  (defun waitx (/ lag speed)
    (setq speed 5)
    (setq lag (+ (getvar "cdate") (* speed 0.000000003)))
    (while (> lag (getvar "cdate")))
  )
  (defun DISPLAY_MSG (/ x1 y1)
    (setq x1 (dimx_tile "set-copyright"))
    (setq y1 (dimy_tile "set-copyright"))
    (start_image "set-copyright")
    (fill_image 0 0 x1 y1 132)
    (end_image)
  )  
  (setq col -15)
  (setq push 12)
  (setq per (rtos 0 2 0))
  (while (< push 600)
    (start_image "set-copyright")
    (fill_image 0 0 push 20 col)
    (end_image)    
    (waitx)
    (setq push (+ push 12))
    (setq per (rtos (+ (atof per) 5) 2 0))    
  )
  (DISPLAY_MSG)
  (set_tile
       "set-copyright"
      (strcat
         "  D i a l o g   a n d   L i s p   D e m o   P r o g r a m     "
         ARCH#YEAR
         "    f o r    A u t o C A D    "
      )
  )  
)

(defun DMO:ABOUT (/ hello1 hello2 hello3 hello4 hello5 hello6 ARCH#LOGO ARCH#YEAR ARCH#YEAR1 ARCH#YEAR2 ARCH#YEAR3 ARCH#YEAR4
                    VAR_LIST VAR_LST VAR)  
  (setq VAR_LST (cdr (cdr GVAR_LIST)))
  (foreach GVAR VAR_LST
    (setq VAR (strcat GVAR "\t" "(" (rtos (car (eval (read GVAR))) 2 0) " " (rtos (cadr (eval (read GVAR))) 2 0) ")"
                            "\t" (substr GVAR 6 (- ( strlen GVAR) 6))
                            "\t" "(" (rtos (car (eval (read (substr GVAR 6 (- ( strlen GVAR) 6))))) 2 0)
                            " " (rtos (cadr (eval (read (substr GVAR 6 (- ( strlen GVAR) 6))))) 2 0) ")"
              )
    )
    (setq VAR_LIST (cons VAR VAR_LIST))
  )
  (setq ARCH#LOGO " About DCL Demo")
  (setq ARCH#YEAR1 (substr (rtos (getvar "CDATE") 2 16) 1 1))
  (setq ARCH#YEAR2 (substr (rtos (getvar "CDATE") 2 16) 2 1))
  (setq ARCH#YEAR3 (substr (rtos (getvar "CDATE") 2 16) 3 1))
  (setq ARCH#YEAR4 (substr (rtos (getvar "CDATE") 2 16) 4 1))
  (setq ARCH#YEAR (strcat ARCH#YEAR1 " " ARCH#YEAR2 " " ARCH#YEAR3 " " ARCH#YEAR4))
  (if (not (new_dialog "DMO_ABOUT" DMO_DCL_ID))(exit))   
  (setq hello1 "MAIN PROGRAM DESIGN AND CODE BY PHILLIP NORMAN")
  (setq hello2 "ORIGINAL \"ABOUT\" DIALOG AND OTHER CODE BY GARY FOWLER") 
  (setq hello3 "") 
  (setq hello4 "Feedback, comments and questions are welcome and encouraged.")
  (setq hello5 "All correspondence to the following email:")
  (setq hello6 "cad@phillipnorman.com")   
  (start_list "dmoinfo")
    (mapcar 'add_list VAR_LIST)
  (end_list)
  (mode_tile "dmoinfo" 1)
  (set_tile
    "set-title"
    (strcat ARCH#LOGO "")
  )        
  (set_tile "hello1" hello1)
  (set_tile "hello2" hello2)
  (set_tile "hello3" hello3)
  (set_tile "hello4" hello4)
  (set_tile "hello5" hello5)
  (set_tile "hello6" hello6)    
  (ANIIT)
  (action_tile "set-title" "(done_dialog)")
  (start_dialog)  
  (princ)
)
; end of Gary Fowler contributions
;
; The following is only for the DCLDemo. It would not form part of a normal program!
; This is where the text for the code window is generated.
(defun dmo:dcl_code (KEY)
  (if (= CODE 1)
      (progn
        (cond
          ((= KEY "ppn_complist")
           (setq CODETXT (list "Action Tile:"
                               "\t(setq COMP# (atoi $value)) (ppn:dcl_compsct)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_compsct ()"
                               "\t  (setq COMPID (list (car COMPID) COMP#))"
                               "\t  (cond"
                               "\t    ((= (car COMPID) 1)"
                               "\t     (setq WFID (list 1 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 2)"                               
                               "\t     (setq WWFID (list 2 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 3)"                               
                               "\t     (setq CHID (list 3 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 4)"
                               "\t     (setq MCHID (list 4 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 5)"
                               "\t     (setq EAID (list 5 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 6)"
                               "\t     (setq SBID (list 6 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 7)"
                               "\t     (setq RHSID (list 7 COMP#))"
                               "\t    )"
                               "\t    ((= (car COMPID) 8)"
                               "\t     (setq CHSID (list 8 COMP#))"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_wf")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 1 \"WFID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_wwf")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 2 \"WWFID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_ch")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 3 \"CHID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_mch")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 4 \"MCHID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_ea")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 5 \"EAID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_sb")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 6 \"SBID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_rhs")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 7 \"RHSID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_chs")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_complist 8 \"CHSID\")"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_complist (ID VAR)"
                               "\t  (setq DMO_COMPLIST (ppn:dcl_buildlist_compsct ID UNITS))"
                               "\t  (setq COMPID (list ID (cadr (eval (read VAR)))))"
                               "\t  (start_list \"ppn_complist\")"
                               "\t  (mapcar 'add_list DMO_COMPLIST)"
                               "\t  (end_list)"
                               "\t  (set_tile \"ppn_complist\" (rtos (cadr (eval (read VAR)))))"
                               "\t  (cond"
                               "\t    ((member ID '(1 3 5 7))"
                               "\t     (foreach STR '(\"ppn_wwf\" \"ppn_mch\" \"ppn_sb\" \"ppn_chs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t    ((member ID '(2 4 6 8))"
                               "\t     (foreach STR '(\ppn_wf\" \"ppn_ch\" \"ppn_ea\" \"ppn_rhs\")"
                               "\t       (set_tile STR \"0\")"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_view")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_view $value 'COMPOP)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_view (VALUE VAR)"
                               "\t  (cond"
                               "\t    ((= VALUE \"0\")\t;first item in list is item \"0\". In this case Section"
                               "\t     (mode_tile \"ppn_hatch\" 0)\t;turn on the hatch toggle"
                               "\t     (mode_tile \"ppn_hide\" 1)\t;turn off the hide lines button"
                               "\t     (mode_tile \"ppn_1stend\" 1)\t;turn off the elevation end line toggle"
                               "\t     (mode_tile \"ppn_2ndend\" 1)\t;turn off the elevation end line toggle"
                               "\t    )"
                               "\t    ((= VALUE \"1\")\t;as above but for Plan"
                               "\t     (mode_tile \"ppn_hatch\" 1)"
                               "\t     (mode_tile \"ppn_hide\" 0)"
                               "\t     (mode_tile \"ppn_1stend\" 1)"
                               "\t     (mode_tile \"ppn_2ndend\" 1)"
                               "\t    )"
                               "\t    ((= VALUE \"2\")\t;as above but for Elevation"
                               "\t     (mode_tile \"ppn_hatch\" 1)"
                               "\t     (mode_tile \"ppn_hide\" 0)"
                               "\t     (mode_tile \"ppn_1stend\" 0)"
                               "\t     (mode_tile \"ppn_2ndend\" 0)"
                               "\t    )"
                               "\t  )"
                               "\t  (ppn:dcl_options \"VIEW\" VALUE VAR)\t;reset the local options variable \"VIEW\" data"
                               "\t)"
                         )
           )
          )
;          ((= KEY "ppn_rotang")
;           (setq CODETXT (list "Action Tile:"
;                               "\t(setq ROTANG (list $value))"
;                               ""
;                               "Action Function:"
;                               "\tnone"
;                         )
;           )
;          )
          ((= KEY "ppn_hide")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_options \"HIDE\" $value 'COMPOP)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_options (KEY VALUE VAR)"
                               "\t  (set VAR (subst (cons KEY VALUE) (assoc KEY (eval VAR)) (eval VAR)))"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_hatch")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_options \"HATCH\" $value 'COMPOP)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_options (KEY VALUE VAR)"
                               "\t  (set VAR (subst (cons KEY VALUE) (assoc KEY (eval VAR)) (eval VAR)))"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_1stend")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_ends $key $value 'COMPOP)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_ends (KEY VALUE VAR)"
                               "\t  (cond\t;cond evaluates only the first true expression"
                               "\t    ((= KEY \"ppn_1stend\")\t;if the toggle picked was the 1stend toggle"
                               "\t     (if (= VALUE \"1\");if it was turned on"
                               "\t       (setq VALUE (1+ (atoi (cdr (assoc '\"ENDS\" COMPOP)))))\t;add 1 to the current value of the \"ENDS\" option"
                               "\t       (setq VALUE (1- (atoi (cdr (assoc '\"ENDS\" COMPOP)))))\t;else subtract 1 from the current value"
                               "\t     )"
                               "\t    )"
                               "\t    ((= KEY \"ppn_2ndend\")\t;if the toggle picked was the 2ndend toggle"
                               "\t     (if (= VALUE \"1\")\t;as above..."
                               "\t       (setq VALUE (+ (atoi (cdr (assoc '\"ENDS\" COMPOP))) 2))"
                               "\t       (setq VALUE (- (atoi (cdr (assoc '\"ENDS\" COMPOP))) 2))"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t  (ppn:dcl_options \"ENDS\" (itoa VALUE) VAR)\t;reset the local options variable \"ENDS\" data"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_2ndend")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_ends $key $value 'COMPOP)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_ends (KEY VALUE VAR)"
                               "\t  (cond\t;cond evaluates only the first true expression"
                               "\t    ((= KEY \"ppn_1stend\")\t;if the toggle picked was the 1stend toggle"
                               "\t     (if (= VALUE \"1\");if it was turned on"
                               "\t       (setq VALUE (1+ (atoi (cdr (assoc '\"ENDS\" COMPOP)))))\t;add 1 to the current value of the \"ENDS\" option"
                               "\t       (setq VALUE (1- (atoi (cdr (assoc '\"ENDS\" COMPOP)))))\t;else subtract 1 from the current value"
                               "\t     )"
                               "\t    )"
                               "\t    ((= KEY \"ppn_2ndend\")\t;if the toggle picked was the 2ndend toggle"
                               "\t     (if (= VALUE \"1\")\t;as above..."
                               "\t       (setq VALUE (+ (atoi (cdr (assoc '\"ENDS\" COMPOP))) 2))"
                               "\t       (setq VALUE (- (atoi (cdr (assoc '\"ENDS\" COMPOP))) 2))"
                               "\t     )"
                               "\t    )"
                               "\t  )"
                               "\t  (ppn:dcl_options \"ENDS\" (itoa VALUE) VAR)\t;reset the local options variable \"ENDS\" data"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_graphscr")
           (setq CODETXT (list "Action Tile:"
                               "\t(setq DCLPOS (done_dialog 2))"
                               ""
                               "Action Function:"
                               "\t(setq WHAT_NEXT (start_dialog))\t;start the dialog box. Values returned by done_dialog are stored in WHAT_NEXT"
                               "\t(if (= WHAT_NEXT 2)\t;if statement to call function after Graphics Screen button is pressed"
                               "\t    (ppn:dcl_graphscr)"
                               "\t)"
                               "\t(defun ppn:dcl_graphscr ()"
                               "\t  (setq INSP (getpoint \"\nSelect an insertion point: \"))"
                               "\t(princ)"
                               "\t)"
                         )
           )
          )
          ((= KEY "ppn_repos")
           (setq CODETXT (list "Action Tile:"
                               "\t(done_dialog 3)(setq DCLPOS '(-1 -1))"
                               ""
                               "Action Function:"
                               "\tThere is no other action associated with this button. By calling (done_dialog 3)"
                               "\tthe dialog is closed. The \"3\" tells the while loop to go back to the"
                               "\tbeginning and by setting the variable DCLPOS to '(-1 -1) the dialog box is"
                               "\tre-drawn, with all the same settings, back in the middle of the screen."
                         )
           )
          )
          ((= KEY "ppn_about")
           (setq CODETXT (list "Action Tile:"
                               "\t(DMO:ABOUT)"
                               ""
                               "Action Function:"
                               "\tRefer to the lisp file."
                         )
           )
          )
          ((= KEY "ppn_code")
           (setq CODETXT (list "Action Tile:"
                               "\t(ppn:dcl_code)"
                               ""
                               "Action Function:"
                               "\t(defun ppn:dcl_code ()"
                               "\t  (setq CODE (boole 6 CODE 1))"
                               "\t  (if (= CODE 1)"
                               "\t    (setq DCLNAME \"DCLDEMO_CODE\")"
                               "\t    (setq DCLNAME \"DCLDEMO\")"
                               "\t  )"
                               "\t  (setq DCLPOS (done_dialog 4))"
                               "\t)"
                         )
           )
          )
        )
      )
  )
  (start_list "ppn_codetxt")
  (mapcar 'add_list CODETXT)
  (end_list)
)

(princ)
(princ "\n*** DCL and Lisp Demo Loaded ***\nEnter PPN_DCLDEMO to run.")
